//
//  TodoListEditor.swift
//  Do It
//
//  Created by Jim Dovey on 2/4/20.
//  Copyright © 2020 Jim Dovey. All rights reserved.
//

import SwiftUI

struct TodoListEditor: View {
    // START:PresentationEnvironment
    @Environment(\.presentationMode) var presentation
    // END:PresentationEnvironment
    // START:BasicProperties
    @EnvironmentObject var data: DataCenter
    @State var list: TodoItemList
    // END:BasicProperties

    // START:IconGradient
    var iconGradient: LinearGradient {
        var (h, s, b) = list.color.hsb
        b *= 0.7
        
        return LinearGradient(
            gradient: Gradient(colors: [
                list.color.uiColor,
                Color(hue: h, saturation: s, brightness: b)
            ]),
            startPoint: .topLeading,
            endPoint: .bottom)
    }
    // END:IconGradient

    // START:BodyStructure
    var body: some View {
        VStack {
            // <literal:elide> Top bar: cancel, title, done </literal:elide>
            // END:BodyStructure
            // START:TopBar
            HStack(alignment: .firstTextBaseline) {
                // START:ButtonContent
                Button("Cancel") {
                    // END:ButtonContent
                    // <literal:elide> dismiss sheet </literal:elide>
                    // END:TopBar
                    // START:ButtonContent
                    // START_HIGHLIGHT
                    self.presentation.wrappedValue.dismiss()
                    // END_HIGHLIGHT
                    // START:TopBar
                }
                Spacer()
                Text("Name & Appearance")
                    .bold()
                Spacer()
                Button(action: {
                    // END:ButtonContent
                    // <literal:elide> save data, dismiss sheet </literal:elide>
                    // START:ButtonContent
                    // END:TopBar
                    // START_HIGHLIGHT
                    self.saveData()
                    self.presentation.wrappedValue.dismiss()
                    // END_HIGHLIGHT
                    // START:TopBar
                }) {
                    Text("Done")
                        .bold()
                }
                // END:ButtonContent
            }
            .padding()
            // END:TopBar
            // START:BodyStructure

            // <literal:elide> List icon </literal:elide>
            // END:BodyStructure
            // START:ListIcon
            Image(systemName: list.icon)
                .font(.system(size: 56, weight: .bold, design: .rounded))
                .aspectRatio(contentMode: .fit)
                .padding(36)
                .foregroundColor(.white)
                .frame(width: 112, height: 112)
                .background(iconGradient)
                .clipShape(Circle())
                .modifier(DoubleShadow())
            // END:ListIcon
            // START:BodyStructure

            // <literal:elide> Text field </literal:elide>
            // END:BodyStructure
            // START:NameField
            TextField("List Title", text: $list.name)
                .font(.system(size: 20, weight: .semibold, design: .rounded))
                .modifier(BorderedTextField())
                .padding()
            // END:NameField
            // START:BodyStructure
            
            VStack(spacing: 0) {
                Divider()
                // START:AddColorPicker
                ScrollView {
                    // END:AddColorPicker
                    // <literal:elide> Color picker </literal:elide>
                    // <literal:elide> Icon chooser </literal:elide>
                    // END:BodyStructure
                    // START:AddColorPicker
                    // START_HIGHLIGHT
                    ColorPicker(selectedColor: $list.color)
                        .padding()
                    IconChooser(selectedIcon: $list.icon)
                        .padding()
                    // END_HIGHLIGHT
                    // START:BodyStructure
                }
                // END:AddColorPicker
            }
        }
    }
    // END:BodyStructure

    // START:SaveData
    private func saveData() {
        if let idx = data.todoLists.firstIndex(where: { $0.id == list.id }) {
            data.todoLists[idx] = list
        }
    }
    // END:SaveData
}

// START:Preview
struct TodoListEditor_Previews: PreviewProvider {
    static var previews: some View {
        TodoListEditor(list: defaultTodoLists[3])
            .environmentObject(DataCenter())
            .padding(.top, 30)
    }
}
// END:Preview
